// Idempotent, modular header.js
// - Exporterar window.headerHTML (markup + CSS)
// - Exporterar window.initializeHeader(root?) som bara binder logik EN gång
// - INGEN auto-injektion och INGEN auto-init på DOMContentLoaded

console.log('header.js laddad (modul-läge)');

// === MARKUP + STYLES ===
const headerHTML = `
<style>
    /* --- Befintliga CSS-regler --- */
    :root {
        --primary-green: #2ecc71; --dark-blue: #34495e; --medium-gray: #ecf0f1; --light-gray-bg: #f4f7f6;
        --border-color: #dce0e3; --text-color: var(--dark-blue); --text-color-light: #7f8c8d;
        --white: #ffffff; --danger: #dc3545; --shadow: 0 4px 6px rgba(0, 0, 0, 0.08);
        --header-height: 64px; --bottom-bar-height: 70px;
        font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, "Helvetica Neue", Arial, sans-serif;
    }
    * { box-sizing: border-box; }
    .app-header { background-color: var(--white); border-bottom: 1px solid var(--border-color); box-shadow: var(--shadow); display: flex; align-items: center; justify-content: space-between; height: var(--header-height); padding: 0 24px; position: sticky; top: 0; z-index: 999; }
    .header-left, .header-center, .header-right { display: flex; align-items: center; gap: 24px; }
    .header-left { flex: 1; justify-content: flex-start; } .header-center { flex: 1; justify-content: center; } .header-right { flex: 1; justify-content: flex-end; }
    .logo a { display: flex; align-items: center; height: calc(var(--header-height) - 24px); }
    .logo img { max-height: 100%; width: auto; display: block; }
    .main-nav ul { margin: 0; padding: 0; list-style: none; display: flex; gap: 8px; background-color: var(--medium-gray); border-radius: 8px; padding: 4px; }
    .main-nav a { text-decoration: none; color: var(--text-color); padding: 8px 12px; border-radius: 6px; font-weight: 600; white-space: nowrap; min-width: 80px; text-align: center; display: block; }
    .main-nav a:hover { background-color: var(--border-color); }
    .main-nav a.active { background-color: var(--white); box-shadow: 0 2px 4px rgba(0, 0, 0, 0.1); color: var(--dark-blue); }
    .btn-expert { background-color: var(--primary-green); color: var(--white); border: none; padding: 8px 18px; border-radius: 9999px; cursor: pointer; font-weight: 700; box-shadow: 0 2px 4px rgba(0,0,0,0.1); font-family: inherit; font-size: 14px; }
    .btn-expert:hover { background-color: #28a745; transform: translateY(-1px); }
    .icon-wrapper { position: relative; }
    .icon-button { background: none; border: none; cursor: pointer; border-radius: 8px; width: 40px; height: 40px; transition: background-color 0.2s ease; }
    .icon-button:hover { background-color: var(--medium-gray); }
    .icon-button svg { width: 24px; height: 24px; stroke: var(--text-color-light); }
    .avatar { width: 40px; height: 40px; border-radius: 50%; cursor: pointer; border: 2px solid transparent; transition: border-color 0.2s ease; }
    .avatar:hover { border-color: var(--primary-green); }
    .notification-badge { position: absolute; top: 4px; right: 4px; background: #e74c3c; color: white; border-radius: 9999px; font-size: 10px; padding: 2px 6px; display: flex; align-items: center; justify-content: center; line-height: 1; border: 2px solid var(--white); }
    .dropdown-panel { display: none; position: absolute; top: calc(100% + 8px); right: 0; width: 300px; background: var(--white); border: 1px solid var(--border-color); box-shadow: var(--shadow); border-radius: 12px; overflow: hidden; transform-origin: top right; animation: fadeInScale 0.2s ease-out; }
    .dropdown-panel.show { display: block; }
    @keyframes fadeInScale { from { opacity: 0; transform: scale(0.98) translateY(-5px); } to { opacity: 1; transform: scale(1) translateY(0); } }
    .dropdown-header { padding: 12px 16px; font-weight: bold; border-bottom: 1px solid var(--border-color); color: var(--dark-blue); }
    .dropdown-content { padding: 8px; max-height: 300px; overflow-y: auto; }
    .dropdown-content ul { list-style: none; padding: 0; margin: 0; }
    .dropdown-content a { display: block; padding: 10px 12px; text-decoration: none; color: var(--text-color); border-radius: 6px; }
    .dropdown-content a:hover { background-color: var(--light-gray-bg); }
    .avatar-menu .divider { height: 1px; background-color: var(--border-color); margin: 6px 0; }
    .notification-item { padding: 12px; border-bottom: 1px solid var(--border-color); font-size: 0.9em; line-height: 1.4; }
    .notification-item:last-child { border-bottom: none; }
	.bottom-tab-bar .icon { 
	  stroke: currentColor; 
	  fill: none;
	}
	.notification-badge {
	  pointer-events: none; /* låt klick gå igenom till #notifications-toggle */
	}


    /* --- NYA STILAR FÖR MOBIL & BOTTEN-BAR --- */
    .bottom-tab-bar {
        display: none; /* Dold på desktop som standard */
    }

    @media (max-width: 800px) {
        /* --- Justeringar för headern på mobil --- */
        .app-header { padding: 0 16px; }
        .main-nav { display: none; }
        .header-left .logo a { height: 40px; }
        .header-right { gap: 12px; }
        .btn-expert { display: none; }

        /* --- Bottom tab bar synlig på mobil --- */
        .bottom-tab-bar {
            position: fixed;
            left: 0; right: 0; bottom: 0;
            height: var(--bottom-bar-height);
            background: var(--white);
            border-top: 1px solid var(--border-color);
            box-shadow: 0 -2px 6px rgba(0,0,0,0.06);
            display: grid;
            grid-template-columns: repeat(4, 1fr);
            align-items: center;
            z-index: 1000;
        }
        .bottom-tab-bar a {
            display: flex;
            flex-direction: column;
            align-items: center;
            justify-content: center;
            text-decoration: none;
            color: var(--text-color);
            font-size: 12px;
            gap: 4px;
            height: 100%;
        }
        .bottom-tab-bar a.active { font-weight: 700; }
        .bottom-tab-bar .icon { width: 22px; height: 22px; }
        body { padding-bottom: var(--bottom-bar-height); } /* undvik att innehåll hamnar bakom baren */
    }
</style>

<header class="app-header">
    <div class="header-left">
        <div class="logo">
            <a href="../prototyp_hem/hem_3_header-injected.html" aria-label="Gå till startsidan">
                <img src="https://www.gomonday.se/main-logo.svg" alt="Logotyp">
            </a>
        </div>
    </div>

    <div class="header-center">
        <nav class="main-nav" aria-label="Huvudnavigering">
            <ul>
                <li><a href="../prototyp_hem/hem_3_header-injected.html" data-navkey="hem">Hem</a></li>
                <li><a href="#" data-navkey="utforska">Utforska</a></li>
                <li><a href="../prototyp_skapa/skapa_7_header-injected.html" data-navkey="skapa">Mitt material</a></li>
            </ul>
        </nav>
    </div>

    <div class="header-right">
        <button class="btn-expert">Experthjälp</button>
        <div class="icon-wrapper">
            <button class="icon-button" id="notifications-toggle" aria-label="Notifieringar">
                <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="1.5" d="M14.857 17.082a23.848 23.848 0 005.454-1.31A8.967 8.967 0 0118 9.75V9a6 6 0 10-12 0v.75a8.967 8.967 0 01-2.312 6.02c1.733.64 3.56 1.085 5.454 1.31m5.715 0a24.255 24.255 0 01-5.714 0m5.714 0a3 3 0 11-5.714 0" /></svg>
            </button>
            <div class="notification-badge" id="notification-badge">1</div>
            <div id="notifications-panel" class="dropdown-panel">
                <div class="dropdown-header">Notifieringar</div>
                <div class="dropdown-content">
                  <div class="notification-item">Påminnelse: Du har ett samtal bokat imorgon kl. 10.00.</div>
                </div>
            </div>
        </div>
        <div class="icon-wrapper">
            <img src="https://i.pravatar.cc/40" alt="Användaravatar" class="avatar" id="avatar-toggle">
            <div id="avatar-menu" class="dropdown-panel">
                 <div class="dropdown-header">Min Profil</div>
                 <div class="dropdown-content">
                    <ul class="avatar-menu">
                        <li><a href="#">Inställningar</a></li>
                        <li><a href="#">Fakturor</a></li>
                        <li><a href="#">Fråga oss</a></li>
                        <li><a href="#">Aktivera partnerkod</a></li>
                        <li class="divider"></li>
                        <li><a href="#">Logga ut</a></li>
                    </ul>
                 </div>
            </div>
        </div>
    </div>
</header>

<nav class="bottom-tab-bar">
    <a href="../prototyp_hem/hem_3_header-injected.html" data-navkey="hem">
        <svg class="icon" xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="1.5" d="M2.25 12l8.954-8.955a1.125 1.125 0 011.591 0L21.75 12M4.5 9.75v10.125c0 .621.504 1.125 1.125 1.125H9.75V15a2.25 2.25 0 012.25-2.25h0A2.25 2.25 0 0114.25 15v6h4.125c.621 0 1.125-.504 1.125-1.125V9.75M8.25 21h7.5" /></svg>
        <span>Hem</span>
    </a>
    <a href="#" data-navkey="utforska">
        <svg class="icon" xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="1.5" d="M6.115 5.19l3.143 4.886c.101.157.13.351.076.533l-1.16 3.867a1.125 1.125 0 001.161.886l.143.048a1.125 1.125 0 00.868.495l.55 1.1" /></svg>
        <span>Utforska</span>
    </a>
    <a href="../prototyp_skapa/skapa_7_header-injected.html" data-navkey="skapa">
        <svg class="icon" xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="1.5" d="M3.375 19.5h17.25M3.375 4.5h17.25M3.375 9.125h17.25M3.375 14.875h17.25M6 9.125h12M6 14.875h7.5" /></svg>
        <span>Mitt material</span>
    </a>
    <a href="#" data-navkey="expert">
        <svg class="icon" xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="1.5" d="M15.75 6a3.75 3.75 0 11-7.5 0 3.75 3.75 0 017.5 0zM4.5 20.25a8.25 8.25 0 1115 0" /></svg>
        <span>Experthjälp</span>
    </a>
</nav>
`;

// === INITIALISERING (idempotent) ===
(function() {
  // Exponera markup globalt
  window.headerHTML = headerHTML;

  // Idempotensflagga för init
  let INIT_DONE = false;

  // Gör init med frivillig root (t.ex. #header-placeholder). Om den saknas jobbar vi mot document.
  window.initializeHeader = function initializeHeader(root) {
    if (INIT_DONE) return; // redan initierad
    INIT_DONE = true;

    // --- Logik för dropdown-menyer ---
    const notificationsToggle = document.getElementById('notifications-toggle');
    const notificationsPanel = document.getElementById('notifications-panel');
    const avatarToggle = document.getElementById('avatar-toggle');
    const avatarMenu = document.getElementById('avatar-menu');

    function closeAll() {
        notificationsPanel?.classList.remove('show');
        avatarMenu?.classList.remove('show');
    }

    notificationsToggle?.addEventListener('click', (e) => {
        e.stopPropagation();
        avatarMenu?.classList.remove('show');
        notificationsPanel?.classList.toggle('show');
    });

    avatarToggle?.addEventListener('click', (e) => {
        e.stopPropagation();
        notificationsPanel?.classList.remove('show');
        avatarMenu?.classList.toggle('show');
    });

    document.addEventListener('click', (e) => {
        if (!e.target.closest('.icon-wrapper')) closeAll();
    });

    // --- Logik för aktiv nav-länk (fungerar för både desktop och mobil) ---
    const navLinks = document.querySelectorAll('a[data-navkey]');
    const currentPagePath = window.location.pathname.toLowerCase();
    
    navLinks.forEach(link => {
        const navKey = link.dataset.navkey;
        if (link.dataset.navkey && (currentPagePath.includes(`/${navKey}_`) || currentPagePath.includes(navKey + ".html"))) {
            link.classList.add('active');
        }
    });
  }; // end initializeHeader
})(); // end IIFE

// OBS! Ingen auto-montering och ingen DOMContentLoaded-lyssnare här.
// Sidan ansvarar för:
// 1) att stoppa in window.headerHTML i önskat element
// 2) att sedan anropa window.initializeHeader(...)
