/**
 * Complete script.js for Candidate Dashboard Mockup
 * - Uses Supabase for data persistence.
 * - Implements persistent drag-and-drop reordering via 'item_order' column (in Column View).
 * - Assumes RLS policies are set up in Supabase allowing anon insert/select/update/delete.
 * - Assumes 'items' table exists with columns: id (uuid, PK), created_at, text (NOT NULL), category (NOT NULL), tab_id (NOT NULL), item_order (integer).
 * - Implements responsiveness adjustments.
 * - Cards View shows top 2 items per category.
 * - Slideshow View cycles through different themes.
 * - Fixed Dump modal close button bug.
 */
document.addEventListener('DOMContentLoaded', () => {
    console.log("DOM fully loaded and parsed");

    try {
        // --- Supabase Setup ---
        const SUPABASE_URL = 'https://yauoxnbusxheigudvdva.supabase.co'; // Your Supabase URL
        const SUPABASE_ANON_KEY = 'eyJhbGciOiJIUzI1NiIsInR5cCI6IkpXVCJ9.eyJpc3MiOiJzdXBhYmFzZSIsInJlZiI6InlhdW94bmJ1c3hoZWlndWR2ZHZhIiwicm9sZSI6ImFub24iLCJpYXQiOjE3NDI5ODI4MjUsImV4cCI6MjA1ODU1ODgyNX0.EZ-qq0qQYHmdM4HrF0ujlyEiiwQtL9w19kjVu5wFaS4'; // Your Supabase Anon Key

        if (!window.supabase) {
            console.error("Supabase client not loaded. Ensure the CDN script is included in index.html.");
            alert("Error: Application cannot connect to the database.");
            return;
        }
        console.log("Supabase client found.");

        const { createClient } = window.supabase;
        const supabase = createClient(SUPABASE_URL, SUPABASE_ANON_KEY);
        console.log("Supabase client created.");

        // --- Constants ---
        const ITEMS_TABLE = 'items';
        const SLIDESHOW_THEMES = ['slide-theme-1', 'slide-theme-2', 'slide-theme-3', 'slide-theme-4']; // CSS classes for slide themes

        // --- DOM Elements ---
        const tabLinks = document.querySelectorAll('.tab-link');
        const tabContents = document.querySelectorAll('.tab-content');
        const mainContainer = document.querySelector('main');
        const body = document.body;
        const itemModal = document.getElementById('item-modal');
        const modalTitle = document.getElementById('modal-title');
        const modalInput = document.getElementById('modal-input');
        const modalItemId = document.getElementById('modal-item-id');
        const modalTargetTab = document.getElementById('modal-target-tab');
        const modalTargetCategory = document.getElementById('modal-target-category');
        const modalSaveBtn = document.getElementById('modal-save-btn');
        const allModalCloseBtns = document.querySelectorAll('[data-modal-target]'); // Generic selector for all modal close triggers
        const dumpModal = document.getElementById('dump-modal');
        const dumpBtn = document.getElementById('dump-btn');
        const dumpInput = document.getElementById('dump-input');
        const guidedSidebar = document.getElementById('guided-sidebar');
        const guidedActionsList = document.getElementById('guided-actions-list');
        const closeSidebarBtn = document.getElementById('close-sidebar-btn');
        const modeSwitches = document.querySelectorAll('input[name="mode"]');

        console.log("DOM elements selected.");

        // --- Data Management (Using Supabase) ---

        let appData = getInitialDataStructure();
        console.log("Initial data structure created:", appData);

        // Function to fetch ALL items and structure them for rendering
        async function loadData() {
            console.log("Fetching data from Supabase...");
            const { data, error } = await supabase
                .from(ITEMS_TABLE)
                .select('*')
                .order('item_order', { ascending: true, nullsFirst: true }) // Primary sort by explicit order
                .order('created_at', { ascending: true }); // Secondary sort for items without order

            if (error) {
                console.error('Error fetching data:', error);
                alert(`Error fetching data: ${error.message}`);
                return getInitialDataStructure();
            }

            console.log("Data fetched successfully:", data);
            appData = getInitialDataStructure(); // Reset local structure

            data.forEach(item => {
                if (appData[item.tab_id] && appData[item.tab_id][item.category]) {
                    if (item.id && typeof item.text === 'string') {
                        appData[item.tab_id][item.category].push({
                            id: item.id,
                            text: item.text,
                            item_order: item.item_order // Ensure item_order is stored
                        });
                    } else {
                        console.warn("Skipping invalid item from DB (missing id or text):", item);
                    }
                } else {
                    console.warn(`Item category or tab_id mismatch or not defined in initial structure: tab_id=${item.tab_id}, category=${item.category}. Item:`, item);
                }
            });

             // Ensure all categories within appData are sorted by item_order after loading
             // This is crucial for Cards view (top 2) and consistent rendering
             for (const tabId in appData) {
                 for (const category in appData[tabId]) {
                     appData[tabId][category].sort((a, b) => {
                        const orderA = a.item_order ?? Infinity;
                        const orderB = b.item_order ?? Infinity;
                        return orderA - orderB;
                     });
                 }
             }
            console.log("Data processed and sorted into appData:", appData);
            return appData;
        }

        // Gets the basic { tabId: { category: [] } } structure
        function getInitialDataStructure() {
            const structure = {};
            tabContents.forEach(tab => {
                const tabId = tab.dataset.tabId;
                if (!tabId) {
                    console.warn("Tab content missing data-tab-id:", tab);
                    return;
                }
                structure[tabId] = {};
                const tools = tab.querySelectorAll('.tool-card');
                tools.forEach(tool => {
                    const category = tool.dataset.category;
                    if (category) {
                        structure[tabId][category] = [];
                    } else {
                         console.warn("Tool card missing data-category:", tool);
                    }
                });
            });
             // Ensure known categories exist even if tool cards change
            const knownCategories = {
                candidateProfile: ['strengths', 'drivers', 'skills'],
                myOpportunities: ['roles', 'industries', 'environment'],
                goalStrategies: ['tracks', 'strategies'],
                documents: ['cv', 'coverLetter', 'linkedIn']
            };
            Object.keys(knownCategories).forEach(tabId => {
                if (structure[tabId]) {
                    knownCategories[tabId].forEach(cat => {
                        if (!structure[tabId][cat]) {
                            console.log(`Manually ensuring category ${cat} exists for tab ${tabId}`);
                            structure[tabId][cat] = [];
                        }
                    });
                } else {
                    console.warn(`Tab ID ${tabId} from knownCategories not found in HTML structure.`);
                }
            });
            return structure;
        }

        // Add a new item to Supabase
        async function addItem(tabId, category, text) {
            console.log(`Adding item: tab=${tabId}, cat=${category}, text=${text}`);
             if (!tabId || !category || !text) {
                 console.error("addItem validation failed:", { tabId, category, text });
                 alert("Cannot add item: Missing required information.");
                 return;
             }
             // Calculate next item_order safely
             const currentItems = appData[tabId]?.[category] || [];
             // Find the highest existing order, default to -1 if empty
             const maxOrder = currentItems.length > 0
                 ? Math.max(...currentItems.map(i => i.item_order ?? -1))
                 : -1;
             const nextOrder = maxOrder + 1;

            const newItem = {
                tab_id: tabId,
                category: category,
                text: text.trim(),
                item_order: nextOrder // Assign the calculated order
            };
            console.log("New item payload:", newItem);

            const { data, error } = await supabase
                .from(ITEMS_TABLE)
                .insert([newItem])
                .select();

            if (error) {
                console.error('Error adding item:', error);
                if (error.message.includes('violates row-level security policy')) {
                     alert('Error adding item: Permission denied. Please check Supabase RLS policies.');
                } else if (error.message.includes('violates not-null constraint')) {
                     alert(`Error adding item: A required field was missing. Details: ${error.message}`);
                } else {
                     alert(`Error adding item: ${error.message}`);
                }
                return;
            }

            console.log("Item added successfully, reloading data:", data);
            await loadDataAndRender();
        }

        // Update an existing item's text in Supabase
        async function updateItem(itemId, newText) {
            console.log(`Updating item: id=${itemId}, text=${newText}`);
            if (!itemId || typeof newText !== 'string') {
                console.error("updateItem validation failed:", {itemId, newText});
                alert("Cannot update item: Missing ID or text.");
                return;
            }

            const { data, error } = await supabase
                .from(ITEMS_TABLE)
                .update({ text: newText.trim() }) // Only update text field
                .eq('id', itemId)
                .select();

            if (error) {
                console.error('Error updating item:', error);
                 if (error.message.includes('violates row-level security policy')) {
                     alert('Error updating item: Permission denied. Check RLS policies.');
                 } else {
                     alert(`Error updating item: ${error.message}`);
                 }
                return;
            }

            console.log("Item updated successfully:", data);
            await loadDataAndRender();
        }

        // Delete an item from Supabase
        async function deleteItem(itemId) {
            if (!itemId) {
                 console.error("deleteItem validation failed: Missing ID");
                 alert("Cannot delete item: Missing ID.");
                 return;
            }
            let itemText = 'this item';
             // Find item details efficiently
            const itemInfo = findItemDetails(itemId);
            if(itemInfo) itemText = `"${itemInfo.item.text}"`;

            if (confirm(`Are you sure you want to delete ${itemText}?`)) {
                console.log(`Deleting item: id=${itemId}`);
                const { error } = await supabase
                    .from(ITEMS_TABLE)
                    .delete()
                    .eq('id', itemId);

                if (error) {
                    console.error('Error deleting item:', error);
                     if (error.message.includes('violates row-level security policy')) {
                         alert('Error deleting item: Permission denied. Check RLS policies.');
                     } else {
                        alert(`Error deleting item: ${error.message}`);
                     }
                    return;
                }

                console.log("Item deleted successfully:", itemId);
                 // If deleting, we also need to re-order items that came after it?
                 // Simpler approach: Reload data. Supabase handles deletion. The next loadData will fetch the correct state.
                 // Optional: could update item_order of subsequent items in the same category/tab via a Supabase function or multiple updates.
                await loadDataAndRender();
            } else {
                 console.log("Deletion cancelled by user.");
            }
        }

        // Helper to find item details (tab, category, object) by ID
        function findItemDetails(itemId) {
             for (const tabId in appData) {
                 for (const category in appData[tabId]) {
                     const item = appData[tabId][category]?.find(i => i.id === itemId);
                     if (item) {
                         return { item, tabId, category };
                     }
                 }
             }
             return null;
        }


        // --- Rendering ---

        function renderApp() {
            console.log("Starting renderApp...");
            if (!appData || Object.keys(appData).length === 0) {
                console.warn("appData is empty or invalid during renderApp.");
                // Optionally render an empty state message in all tabs
                 tabContents.forEach(tabElement => {
                     const views = tabElement.querySelectorAll('.main-view-area > .view');
                     views.forEach(viewElement => viewElement.innerHTML = '<p class="empty-view-message">No data loaded or available.</p>');
                 });
                return; // Exit if no data
            }
            tabContents.forEach(tabElement => {
                const tabId = tabElement.dataset.tabId;
                if (!tabId) {
                     console.warn("Skipping render for tab element missing data-tab-id:", tabElement);
                     return;
                }
                const tabData = appData[tabId];
                if (tabData) {
                    renderAllViews(tabElement, tabData);
                } else {
                    console.warn(`No data found for tab ${tabId} in appData during renderApp. Clearing views.`);
                     const views = tabElement.querySelectorAll('.main-view-area > .view');
                     views.forEach(viewElement => viewElement.innerHTML = '<p class="empty-view-message">No data for this section.</p>');
                }
            });
            const activeTab = document.querySelector('.tab-content.active');
            if (activeTab) {
                initializeTabComponents(activeTab);
            } else {
                 console.warn("No active tab found after rendering.");
            }
             console.log("renderApp finished.");
        }

        function renderAllViews(tabElement, tabData) {
            const views = tabElement.querySelectorAll('.main-view-area > .view');
            views.forEach(viewElement => {
                const viewType = viewElement.dataset.viewId;
                 if (!viewType) {
                     console.warn("Skipping render for view element missing data-view-id:", viewElement);
                     return;
                 }
                const tabId = tabElement.dataset.tabId;
                viewElement.innerHTML = ''; // Clear before rendering
                try {
                    switch (viewType) {
                        case 'columns':
                            renderColumnView(viewElement, tabData, tabId);
                            break;
                        case 'cards':
                            renderCardView(viewElement, tabData, tabId); // Updated view
                            break;
                        case 'slideshow':
                            renderSlideshowView(viewElement, tabData, tabId); // Updated view
                            break;
                        case 'table':
                            renderTableView(viewElement, tabData, tabId);
                            break;
                        default:
                            console.warn(`Unknown view type: ${viewType}`);
                            viewElement.innerHTML = `<p class="error-message">Error: Unknown view type '${viewType}'.</p>`;
                    }
                } catch (renderError) {
                    console.error(`Error rendering view ${viewType} for tab ${tabId}:`, renderError);
                    viewElement.innerHTML = `<p class="error-message">Error rendering this view. Check console.</p>`;
                }
            });
        }

        function createItemControls(itemId) {
            if (!itemId) return '';
            return `
                <div class="item-controls">
                    <button class="edit-btn" data-id="${itemId}" title="Edit"><i class="fas fa-pencil-alt" data-id="${itemId}"></i></button>
                    <button class="delete-btn" data-id="${itemId}" title="Delete"><i class="fas fa-trash-alt" data-id="${itemId}"></i></button>
                </div>
            `;
        }

        // Renders Column view (with drag and drop preparation)
        function renderColumnView(viewElement, tabData, tabId) {
             const toolCards = document.querySelector(`.tab-content[data-tab-id="${tabId}"]`)?.querySelectorAll('.tool-card[data-category]') || [];

             if (toolCards.length === 0) {
                 console.warn(`No tool cards found for tab ${tabId} to render columns.`);
                 viewElement.innerHTML = '<p class="empty-view-message">No categories defined for this tab.</p>';
                 return;
             }

             toolCards.forEach(tool => {
                const category = tool.dataset.category;
                if (!category) return;

                const categoryLabel = getCategoryLabel(category, tabId);
                // Items are pre-sorted by item_order from loadData
                const items = tabData[category] || [];

                const columnDiv = document.createElement('div');
                columnDiv.className = 'column';
                columnDiv.innerHTML = `<h4>${categoryLabel}</h4>`;

                const ul = document.createElement('ul');
                 ul.dataset.category = category;
                 ul.dataset.tabId = tabId;

                if (items.length === 0) {
                    ul.innerHTML = `<li class="empty-column-placeholder">No items yet.</li>`;
                } else {
                    items.forEach(item => {
                        if (!item || !item.id) {
                             console.warn("Skipping rendering invalid item in column view:", item);
                             return;
                        }
                        const li = document.createElement('li');
                        li.draggable = true;
                        li.dataset.id = item.id;
                        li.dataset.text = item.text || ''; // Store text for easier drop handling
                        li.innerHTML = `
                            <span class="item-text">${item.text || '(empty text)'}</span>
                            ${createItemControls(item.id)}
                        `;
                        ul.appendChild(li);
                    });
                }
                columnDiv.appendChild(ul);
                viewElement.appendChild(columnDiv);
             });
             // Add drag/drop listeners after rendering columns for this view
             addDragDropListeners(viewElement);
        }

        // Renders Card view - MODIFIED to show only top 2 per category
        function renderCardView(viewElement, tabData, tabId) {
            let itemCount = 0;
            // Iterate through categories based on the order defined by tool cards if needed, or just object keys
            const toolCategories = Array.from(document.querySelector(`.tab-content[data-tab-id="${tabId}"]`)?.querySelectorAll('.tool-card[data-category]') || [])
                                     .map(tool => tool.dataset.category)
                                     .filter(cat => cat); // Get category order from DOM

            toolCategories.forEach(category => {
                 if (!tabData[category]) return; // Skip if category has no data

                 const categoryLabel = getCategoryLabel(category, tabId);
                 // Items are pre-sorted by item_order in loadData
                 const items = tabData[category] || [];
                 const itemsToShow = items.slice(0, 2); // Get only the first two items

                 itemsToShow.forEach(item => {
                     if (!item || !item.id) {
                         console.warn("Skipping rendering invalid item in card view:", item);
                         return;
                     }
                    const card = document.createElement('div');
                    card.className = 'item-card'; // Class for styling
                    card.dataset.id = item.id;
                    card.innerHTML = `
                        <div class="item-card-content">
                            <span class="category-label">${categoryLabel}:</span>
                            <span class="item-text">${item.text || '(empty text)'}</span>
                        </div>
                        ${createItemControls(item.id)}
                    `;
                    viewElement.appendChild(card);
                    itemCount++;
                });
            });

             if (itemCount === 0) {
                 viewElement.innerHTML = '<p class="empty-view-message">No items to display.</p>';
             }
        }

        // Renders Slideshow view - MODIFIED to apply themes
        function renderSlideshowView(viewElement, tabData, tabId) {
             const container = document.createElement('div');
             container.className = 'slideshow-container';
             container.dataset.slideshowInitialized = "false";

             let slidesData = [];
             let themeIndex = 0; // To cycle through themes

            // Get category order from DOM to render slides in a consistent order
            const toolCategories = Array.from(document.querySelector(`.tab-content[data-tab-id="${tabId}"]`)?.querySelectorAll('.tool-card[data-category]') || [])
                                     .map(tool => tool.dataset.category)
                                     .filter(cat => cat);

             toolCategories.forEach(category => {
                 if (!tabData[category]) return;
                 const categoryLabel = getCategoryLabel(category, tabId);
                 // Items are pre-sorted by item_order in loadData
                 (tabData[category] || []).forEach(item => {
                     if (item && item.id) {
                         const themeClass = SLIDESHOW_THEMES[themeIndex % SLIDESHOW_THEMES.length];
                         slidesData.push({ ...item, categoryLabel, themeClass }); // Add theme class
                         themeIndex++;
                     } else {
                          console.warn("Skipping rendering invalid item in slideshow prep:", item);
                     }
                 });
             });

             slidesData.forEach(item => {
                 const slide = document.createElement('div');
                 // Add base class and specific theme class
                 slide.className = `slide ${item.themeClass}`;
                 slide.dataset.id = item.id;
                 slide.innerHTML = `
                    <div class="slide-content-wrapper">
                         <span class="category-label">${item.categoryLabel}</span>
                         <span class="item-text">${item.text || '(empty text)'}</span>
                    </div>
                    <div class="slide-controls"> ${createItemControls(item.id)} </div>
                 `;
                 container.appendChild(slide);
             });

             if (slidesData.length > 0) {
                  const prevButton = document.createElement('button');
                  prevButton.className = 'slide-nav prev'; // Changed class for clarity
                  prevButton.title = 'Previous';
                  prevButton.innerHTML = '&#10094;';
                  container.appendChild(prevButton);

                  const nextButton = document.createElement('button');
                  nextButton.className = 'slide-nav next'; // Changed class for clarity
                  nextButton.title = 'Next';
                  nextButton.innerHTML = '&#10095;';
                  container.appendChild(nextButton);
             } else {
                 container.innerHTML = '<p class="empty-view-message">No items to display in slideshow.</p>';
             }
             viewElement.appendChild(container);
             // Initialization (event listeners) moved to initializeSlideshow
        }

        // Renders Table view
        function renderTableView(viewElement, tabData, tabId) {
             const table = document.createElement('table');
             table.className = 'data-table';
             table.innerHTML = `
                 <thead>
                     <tr>
                         <th data-sortable data-column="0">Category <span class="sort-icon"></span></th>
                         <th data-sortable data-column="1">Item <span class="sort-icon"></span></th>
                         <th>Actions</th>
                     </tr>
                 </thead>
                 <tbody></tbody>
             `;
             const tbody = table.querySelector('tbody');
             tbody.dataset.tabId = tabId; // Store context for sorting if needed

             let allItems = [];
             // Get category order from DOM
             const toolCategories = Array.from(document.querySelector(`.tab-content[data-tab-id="${tabId}"]`)?.querySelectorAll('.tool-card[data-category]') || [])
                                      .map(tool => tool.dataset.category)
                                      .filter(cat => cat);

             toolCategories.forEach(category => {
                  if (!tabData[category]) return;
                  const categoryLabel = getCategoryLabel(category, tabId);
                 // Items are pre-sorted by item_order in loadData
                 (tabData[category] || []).forEach(item => {
                      if (item && item.id) {
                         allItems.push({ ...item, categoryLabel });
                     } else {
                         console.warn("Skipping rendering invalid item in table prep:", item);
                     }
                 });
             });
            // Initial sort can be done here or rely on table header clicks
             // Let's keep the order from the categories/item_order for initial render
             // allItems.sort(...) // Removed initial sort - rely on pre-sorted data

             allItems.forEach(item => {
                 const row = document.createElement('tr');
                 row.dataset.id = item.id;
                 row.dataset.category = item.category; // Store original category
                 row.innerHTML = `
                     <td>${item.categoryLabel}</td>
                     <td class="item-text">${item.text || '(empty text)'}</td>
                     <td>${createItemControls(item.id)}</td>
                 `;
                 tbody.appendChild(row);
             });

              if (allItems.length === 0) {
                  const row = tbody.insertRow();
                  const cell = row.insertCell();
                  cell.colSpan = 3;
                  cell.innerHTML = '<span class="empty-view-message">No items to display.</span>';
                  cell.style.textAlign = 'center';
             }

             viewElement.appendChild(table);
              // Initialization (sorting listeners) moved to initializeTableSorting
         }

         function getCategoryLabel(category, tabId) {
            if (!category || !tabId) return 'Unknown Category';
            // Try finding label from tool card first
            const toolCard = document.querySelector(`.tab-content[data-tab-id="${tabId}"] .tool-card[data-category="${category}"]`);
            if (toolCard?.dataset.categoryLabel) return toolCard.dataset.categoryLabel;
            // Fallback: Try finding label from guided action list item
            const guidedItem = document.querySelector(`#guided-actions-list li[data-tool-category="${category}"][data-tab-id="${tabId}"]`);
             if (guidedItem?.dataset.categoryLabel) return guidedItem.dataset.categoryLabel;
            // Final fallback: Format the category key itself
            return category.replace(/([A-Z])/g, ' $1').replace(/^./, str => str.toUpperCase());
        }

        // --- Event Listeners Setup ---
        console.log("Setting up event listeners...");

        // Tab Switching
        tabLinks.forEach(link => {
            link.addEventListener('click', () => {
                const targetTabHtmlId = link.getAttribute('data-tab');
                 console.log(`Tab link clicked: ${targetTabHtmlId}`);
                activateTab(targetTabHtmlId);
            });
        });

        function activateTab(targetTabHtmlId) {
                console.log(`Activating tab: ${targetTabHtmlId}`);
                tabLinks.forEach(l => l.classList.remove('active'));
                tabContents.forEach(c => c.classList.remove('active'));

                const newTabLink = document.querySelector(`.tab-link[data-tab="${targetTabHtmlId}"]`);
                const newTabContent = document.getElementById(targetTabHtmlId);

                if(newTabLink) newTabLink.classList.add('active');
                else console.warn(`Could not find tab link for ${targetTabHtmlId}`);

                if(newTabContent) {
                    newTabContent.classList.add('active');
                    console.log(`Tab content ${targetTabHtmlId} activated.`);
                    // Initialize components AFTER the tab is made visible
                    initializeTabComponents(newTabContent);
                } else {
                    console.error(`Could not find tab content element with ID: ${targetTabHtmlId}`);
                }
        }

        // Event delegation for view switching, edits, deletes, tool card clicks (within MAIN content)
        mainContainer.addEventListener('click', (e) => {
             const target = e.target;

             // View Switcher Buttons
             const viewBtn = target.closest('.view-btn[data-view]');
             if (viewBtn) {
                const parentTabContent = viewBtn.closest('.tab-content');
                if (!parentTabContent) {
                     console.error("Could not find parent tab content for view switcher button:", viewBtn);
                     return;
                }
                 const targetViewId = viewBtn.getAttribute('data-view');
                 const tabId = parentTabContent.dataset.tabId;
                 console.log(`View button clicked for tab ${tabId}: ${targetViewId}`);

                 const viewBtns = parentTabContent.querySelectorAll('.view-switcher .view-btn');
                 const views = parentTabContent.querySelectorAll('.main-view-area > .view');

                 viewBtns.forEach(b => b.classList.remove('active'));
                 views.forEach(v => v.classList.remove('active'));

                 viewBtn.classList.add('active');
                 const newView = parentTabContent.querySelector(`.main-view-area > .view[data-view-id="${targetViewId}"]`);
                 if (newView) {
                     newView.classList.add('active');
                     console.log(`View ${targetViewId} activated for tab ${tabId}.`);
                     initializeViewComponents(newView); // Initialize components of the newly active view
                 } else {
                     console.error(`Could not find view element with data-view-id="${targetViewId}" in tab ${tabId}`);
                 }
                 return; // Handled
             }

             // Edit Buttons
             const editBtn = target.closest('.edit-btn[data-id]');
             if (editBtn) {
                 const itemId = editBtn.dataset.id;
                 console.log(`Edit button clicked for item ${itemId}`);
                 const itemInfo = findItemDetails(itemId); // Use helper function
                 if (itemInfo) {
                      openItemModal(itemInfo.tabId, itemInfo.category, itemInfo.item.text, itemInfo.item.id);
                 } else {
                      console.error("Could not find item locally to edit:", itemId, "Current appData:", appData);
                      alert("Error: Could not find item data to edit.");
                 }
                 return; // Handled
             }

             // Delete Buttons
             const deleteBtn = target.closest('.delete-btn[data-id]');
             if (deleteBtn) {
                 const itemId = deleteBtn.dataset.id;
                 console.log(`Delete button clicked for item ${itemId}`);
                 deleteItem(itemId).catch(console.error);
                 return; // Handled
             }

             // Tool Card Clicks
             const toolCard = target.closest('.tool-card[data-category]');
             if (toolCard) {
                const cardTabId = toolCard.closest('.tab-content')?.dataset.tabId;
                const category = toolCard.dataset.category;
                 if (cardTabId && category) {
                     console.log(`Tool card clicked: tab=${cardTabId}, category=${category}`);
                     openItemModal(cardTabId, category); // Open modal to add new item
                 } else {
                      console.warn("Tool card clicked but missing context (tabId or category):", toolCard);
                 }
                 return; // Handled
             }
        });


        // --- Modal Handling (Generic) ---
        function openModal(modalId) {
            const modal = document.getElementById(modalId);
            console.log(`Opening modal: ${modalId}`);
            if (modal) {
                modal.classList.add('show');
                body.classList.add('modal-open');
                // Focus on the first input/textarea if available
                const focusable = modal.querySelector('input, textarea');
                if (focusable) focusable.focus();
            } else {
                 console.error(`Modal element not found: ${modalId}`);
            }
        }

        function closeModal(modalId) {
            const modal = document.getElementById(modalId);
            console.log(`Closing modal: ${modalId}`);
             if (modal) {
                modal.classList.remove('show');
                 // Only remove body class if no other modals are open
                 const anyModalOpen = document.querySelector('.modal.show');
                 if (!anyModalOpen) {
                    body.classList.remove('modal-open');
                 }
                 // Clear item modal specifics if it was the one closed
                 if (modalId === 'item-modal') {
                     if (modalInput) modalInput.value = '';
                     if (modalItemId) modalItemId.value = '';
                     if (modalTargetTab) modalTargetTab.value = '';
                     if (modalTargetCategory) modalTargetCategory.value = '';
                 }
            } else {
                 console.error(`Modal element not found: ${modalId}`);
             }
        }

        // Open Item Modal (Specific setup)
        function openItemModal(tabId, category, currentText = '', itemId = null) {
            console.log(`Setting up item modal: tab=${tabId}, cat=${category}, id=${itemId || 'new'}`);
            if (!tabId || !category) {
                 console.error("Cannot open item modal: Missing tabId or category.");
                 alert("Internal Error: Cannot determine context for item modal.");
                 return;
            }
            const categoryLabel = getCategoryLabel(category, tabId);
            modalTitle.textContent = itemId ? `Edit ${categoryLabel}` : `Add New ${categoryLabel}`;
            modalInput.value = currentText;
            modalItemId.value = itemId || '';
            modalTargetTab.value = tabId;
            modalTargetCategory.value = category;
            openModal('item-modal'); // Use generic open function
        }

        // Open Dump Modal (Specific setup - though none needed now)
        function openDumpModal() {
            // dumpInput.value = ''; // Optionally clear dump input on open
            openModal('dump-modal'); // Use generic open function
        }


        // --- Event Listeners for Modals & Global ---

        // Generic Modal Close Button Listener
        document.addEventListener('click', (e) => {
            // Close button or cancel button inside a modal
             const closeTrigger = e.target.closest('[data-modal-target]');
             if (closeTrigger) {
                const modalId = closeTrigger.dataset.modalTarget;
                if (modalId) {
                     closeModal(modalId);
                     return; // Handled
                 }
             }
             // Clicking outside the modal content
             if (e.target.classList.contains('modal')) {
                 const modalId = e.target.id;
                 if (modalId) {
                     closeModal(modalId);
                     return; // Handled
                 }
             }
        });


        // Item Modal Save Button
        if (modalSaveBtn) {
            modalSaveBtn.addEventListener('click', async () => {
                 console.log("Modal save button clicked.");
                const text = modalInput.value.trim();
                const itemId = modalItemId.value;
                const tabId = modalTargetTab.value;
                const category = modalTargetCategory.value;
                if (!text) {
                    alert('Please enter some text for the item.');
                     modalInput.focus();
                    return;
                }
                if (!tabId || !category) {
                    console.error("Cannot save item: Missing tab or category context in modal hidden fields.");
                    alert('Internal Error: Missing context (tab or category). Cannot save.');
                    return;
                }
                modalSaveBtn.disabled = true;
                modalSaveBtn.textContent = 'Saving...';
                try {
                    if (itemId) {
                        await updateItem(itemId, text);
                    } else {
                        await addItem(tabId, category, text);
                    }
                    closeModal('item-modal'); // Use generic close
                } catch(err) {
                    console.error("Error during save operation (caught in modal handler):", err);
                    // Specific error handled in update/add functions, show generic here
                     alert("An error occurred while saving. Please check the console for details.");
                } finally {
                     modalSaveBtn.disabled = false;
                     modalSaveBtn.textContent = 'Save';
                }
            });
        } else {
            console.error("Modal Save Button (#modal-save-btn) not found!");
        }

        // Dump Button Listener
        if (dumpBtn) dumpBtn.addEventListener('click', openDumpModal);
        else console.error("Dump Button (#dump-btn) not found!");


        // --- Guided Mode Sidebar ---
        function toggleSidebar(show) {
             if (guidedSidebar) {
                 if (show) {
                    console.log("Showing guided sidebar.");
                    guidedSidebar.classList.add('show');
                    // Check screen width for overlay behavior
                    if (window.innerWidth <= 992) { // Matches max-width in CSS for overlay
                        body.classList.add('sidebar-open-overlay'); // Use a specific class for overlay
                    } else {
                        body.classList.add('sidebar-open-push'); // Use class for pushing content
                    }
                 } else {
                     console.log("Hiding guided sidebar.");
                     guidedSidebar.classList.remove('show');
                     body.classList.remove('sidebar-open-overlay', 'sidebar-open-push'); // Remove both classes
                 }
             } else {
                  console.error("Guided sidebar element not found!");
             }
        }

        modeSwitches.forEach(radio => {
             radio.addEventListener('change', (e) => {
                 if (e.target.checked) {
                    const mode = e.target.value;
                    console.log(`Mode changed to: ${mode}`);
                    toggleSidebar(mode === 'guided');
                 }
             });
        });

        if (closeSidebarBtn) closeSidebarBtn.addEventListener('click', () => {
             // Turn off guided mode when closing sidebar via its button
             const sandboxRadio = document.querySelector('input[name="mode"][value="sandbox"]');
             if (sandboxRadio) sandboxRadio.checked = true;
             toggleSidebar(false);
        });

        if (guidedActionsList) {
            guidedActionsList.addEventListener('click', (e) => {
                const li = e.target.closest('li[data-tool-category][data-tab-id]');
                if (li) {
                    const category = li.dataset.toolCategory;
                    const tabId = li.dataset.tabId;
                    const tabHtmlId = document.querySelector(`.tab-content[data-tab-id="${tabId}"]`)?.id;
                    console.log(`Guided action clicked: category=${category}, tabId=${tabId}`);
                    if (tabHtmlId) {
                        activateTab(tabHtmlId); // Switch to the correct tab first
                        // Use setTimeout to ensure tab switch completes and modal opens correctly
                        setTimeout(() => {
                            openItemModal(tabId, category);
                            toggleSidebar(false); // Optionally close sidebar after action
                            const sandboxRadio = document.querySelector('input[name="mode"][value="sandbox"]');
                            if (sandboxRadio) sandboxRadio.checked = true; // Switch mode back
                        }, 50); // Small delay
                    } else {
                        console.error(`Could not find tab HTML ID corresponding to data-tab-id "${tabId}"`);
                        alert("Error: Could not switch to the correct tab for this action.");
                    }
                }
            });
        } else {
             console.error("Guided Actions List element not found!");
        }


        // --- Drag and Drop (Column View Only) ---

        let draggedElement = null;

        // Add listeners only to the specific column view container
        function addDragDropListeners(columnViewContainer) {
            if (!columnViewContainer) return;
            console.log(`Setting up drag/drop for container:`, columnViewContainer);
            // Remove old listeners first to prevent duplicates if re-rendered
            columnViewContainer.querySelectorAll('ul[data-category] li[draggable="true"]').forEach(draggable => {
                draggable.removeEventListener('dragstart', handleDragStart);
                draggable.removeEventListener('dragend', handleDragEnd);
                draggable.addEventListener('dragstart', handleDragStart);
                draggable.addEventListener('dragend', handleDragEnd);
            });
            columnViewContainer.querySelectorAll('ul[data-category]').forEach(zone => {
                zone.removeEventListener('dragover', handleDragOver);
                zone.removeEventListener('dragleave', handleDragLeave);
                zone.removeEventListener('drop', handleDrop);
                zone.addEventListener('dragover', handleDragOver);
                zone.addEventListener('dragleave', handleDragLeave);
                zone.addEventListener('drop', handleDrop);
            });
        }

        function handleDragStart(e) {
            // Ensure it's a direct list item in a column view
             if (e.target.matches && e.target.matches('.view-columns li[draggable="true"][data-id]')) {
                 draggedElement = e.target;
                 console.log("Drag Start:", draggedElement.dataset.id);
                 // Use text/plain for broader compatibility, storing the ID
                 e.dataTransfer.setData('text/plain', draggedElement.dataset.id);
                 e.dataTransfer.effectAllowed = 'move';
                 // Use setTimeout to allow the browser to render the drag image before adding class
                 setTimeout(() => { if (draggedElement) draggedElement.classList.add('dragging'); }, 0);
             } else {
                 e.preventDefault(); // Prevent dragging unintended elements
                 console.warn("Drag start prevented on non-draggable or unexpected element:", e.target);
             }
        }

        function handleDragEnd(e) {
             // Clean up classes regardless of drop success
             if (draggedElement) {
                  console.log("Drag End:", draggedElement.dataset.id);
                 draggedElement.classList.remove('dragging');
             } else {
                 console.log("Drag End (no valid draggedElement tracked).");
             }
             // Remove highlight from all potential drop zones
             document.querySelectorAll('.view-columns ul.drag-over').forEach(el => el.classList.remove('drag-over'));
             draggedElement = null; // Clear the reference
        }

        function handleDragOver(e) {
             // Necessary to allow dropping
             e.preventDefault();
             const dropZone = e.target.closest('.view-columns ul[data-category]');
             if (dropZone && draggedElement && dropZone.contains(draggedElement)) {
                 // Allow drop within the same list or different valid lists
                 e.dataTransfer.dropEffect = 'move';
                 // Add highlight only to the current zone, remove from others
                 document.querySelectorAll('.view-columns ul.drag-over').forEach(el => { if (el !== dropZone) el.classList.remove('drag-over'); });
                 dropZone.classList.add('drag-over');
             } else if (dropZone && draggedElement && !dropZone.contains(draggedElement)) {
                  // Optionally allow dropping into DIFFERENT category lists
                  // For now, let's stick to reordering within the same category/list
                  // To allow cross-category drop, remove the dropZone.contains(draggedElement) check
                  // and adjust handleDrop logic significantly.
                  e.dataTransfer.dropEffect = 'none'; // Disallow drop in other categories currently
                  // console.log("Preventing drop into different category");
             } else {
                 e.dataTransfer.dropEffect = 'none'; // Disallow drop elsewhere
             }
        }

        function handleDragLeave(e) {
            const dropZone = e.target.closest('.view-columns ul[data-category]');
            // Only remove highlight if leaving the zone entirely, not moving over child elements
            if (dropZone && !dropZone.contains(e.relatedTarget)) {
                dropZone.classList.remove('drag-over');
            }
        }

        // Handles the drop event, calculates new order, and updates Supabase
        async function handleDrop(e) {
            e.preventDefault(); // Prevent default browser action
            const dropZone = e.target.closest('.view-columns ul[data-category]');

            // Abort if not a valid drop zone or no element was being dragged
            if (!dropZone || !draggedElement) {
                 console.warn("Drop aborted: Invalid drop zone or no dragged element.", { dropZone, draggedElement });
                 handleDragEnd(); // Clean up
                return;
            }

             // Clean up visual feedback immediately
            dropZone.classList.remove('drag-over');
            if (draggedElement) draggedElement.classList.remove('dragging');

            const tabId = dropZone.dataset.tabId;
            const category = dropZone.dataset.category;
            const droppedItemId = draggedElement.dataset.id;

            if (!category || !tabId || !droppedItemId) {
                console.error("Drop zone or dragged item missing critical data (category, tabId, itemId)! Cannot save order.", {dropZone, draggedElement});
                alert("Error: Cannot save order. Drop context is missing.");
                handleDragEnd(); // Clean up
                return;
            }
             console.log(`Drop detected on category ${category}, tab ${tabId}, item ${droppedItemId}`);

            // Determine where the item was dropped relative to others
            const afterElement = getDragAfterElement(dropZone, e.clientY);

            // Check if the position actually changed
             const originalIndex = Array.from(dropZone.children).indexOf(draggedElement);
             let newIndex;
             if (afterElement === null) {
                 newIndex = dropZone.children.length -1 ; // Will be appended at the end
             } else {
                  newIndex = Array.from(dropZone.children).indexOf(afterElement);
                 // If dropping before the first element, index is 0.
                 // If dropping right after itself, index effectively doesn't change relative to others.
             }

             // Visually move the element in the DOM
             if (afterElement == null) {
                 dropZone.appendChild(draggedElement);
             } else {
                 dropZone.insertBefore(draggedElement, afterElement);
             }


            // Get the NEW order of item IDs from the DOM
            const updatedItemElements = Array.from(dropZone.querySelectorAll('li[data-id]'));
            const updatedItemIdOrder = updatedItemElements.map(el => el.dataset.id);

             // --- Optimization: Only update if order *actually* changed ---
             // This requires comparing the new ID order to the order in appData[tabId][category]
             const originalItemIdOrder = (appData[tabId]?.[category] || []).map(item => item.id);
             if (JSON.stringify(updatedItemIdOrder) === JSON.stringify(originalItemIdOrder)) {
                 console.log("Drop resulted in no actual change of order. No update needed.");
                 handleDragEnd(); // Clean up
                 return;
             }
            // --- End Optimization ---


            // Create the payload for Supabase update
            const updatePayload = updatedItemElements.map((itemElement, index) => {
                const itemId = itemElement.dataset.id;
                if (!itemId) {
                    console.error("Cannot create update payload: List item missing data-id", itemElement);
                    return null; // Skip this problematic item
                }
                // Note: We only need to update id and item_order for upsert based on 'id' conflict
                return {
                    id: itemId,
                    item_order: index // The new index IS the new order
                };
            }).filter(payload => payload !== null); // Filter out any nulls from problematic items

            if (updatePayload.length !== updatedItemElements.length) {
                 console.error("Some items could not be included in the order update payload due to missing IDs. Order update might be incomplete or fail. Forcing reload.");
                 alert("Error processing drop. Refreshing data.");
                 await loadDataAndRender();
                 handleDragEnd();
                 return;
            }
            if (updatePayload.length === 0 && updatedItemElements.length > 0) {
                 console.error("Could not create payload for any items after drop. Aborting DB update and refreshing.");
                 alert("Error processing drop. Please refresh the page.");
                 await loadDataAndRender();
                 handleDragEnd();
                 return;
            }
            if (updatePayload.length === 0) {
                 console.log("Drop zone is empty or resulted in no items, nothing to update.");
                 handleDragEnd();
                 return; // Nothing to do
            }


            // Update Supabase using upsert (updates existing based on ID, sets item_order)
            console.log(`Updating item order in Supabase for tab=${tabId}, category=${category}. Payload:`, updatePayload);
            const { data: updateData, error: updateError } = await supabase
                .from(ITEMS_TABLE)
                .upsert(updatePayload, { onConflict: 'id' }); // Upsert based on the 'id' column

            if (updateError) {
                console.error('Error updating item order:', updateError);
                alert(`Failed to save new order: ${updateError.message}. Reverting visual change and refreshing data.`);
                // Revert visual change is hard, easier to just reload which will fetch correct state
                await loadDataAndRender(); // Reload data on error
            } else {
                console.log('Item order updated successfully in Supabase.');
                 // Update local appData immediately to reflect the change BEFORE the full reload,
                 // makes UI feel slightly snappier if reload takes time.
                 if (appData[tabId] && appData[tabId][category]) {
                    // Create a map for quick lookup
                    const itemMap = new Map(appData[tabId][category].map(item => [item.id, item]));
                    // Rebuild the category array in the new order
                     appData[tabId][category] = updatedItemIdOrder
                        .map((id, index) => {
                             const existingItem = itemMap.get(id);
                             if (existingItem) {
                                 existingItem.item_order = index; // Update local order
                                 return existingItem;
                             }
                             return null; // Should not happen if DOM is based on appData
                         })
                        .filter(item => item !== null);
                     console.log('Local appData updated with new order.');
                 }
                 // Even with local update, a reload ensures full consistency, especially if other users might be making changes.
                 // Consider if the immediate local update is worth the complexity vs just reloading.
                 await loadDataAndRender(); // Still reload for guaranteed consistency
            }
             handleDragEnd(); // Final cleanup
        }

        // Helper function to find the element to insert before during drag-and-drop
        function getDragAfterElement(container, y) {
            // Get all draggable items in the container EXCEPT the one being dragged
            const draggableElements = [...container.querySelectorAll('li[draggable="true"]:not(.dragging)')];

            return draggableElements.reduce((closest, child) => {
                const box = child.getBoundingClientRect();
                // Calculate the vertical midpoint of the element
                const boxMidpointY = box.top + box.height / 2;
                // Calculate the offset: positive if cursor is below midpoint, negative if above
                const offset = y - boxMidpointY;

                // We are looking for the element *immediately* below the cursor (smallest positive offset)
                // OR the first element whose midpoint the cursor is *above* (smallest negative offset among those > -infinity)
                // Let's refine: find the element whose top edge is *just below* the cursor 'y'.
                // Simplified: Find the element where the cursor 'y' is between its top and bottom.
                // If 'y' is above the midpoint, insert before this element.

                 // Offset from the element's *top* edge
                 const offsetFromTop = y - box.top;

                if (offset < 0 && offset > closest.offset) {
                     // If cursor is above the midpoint, this element is a candidate to insert before.
                     // We want the one with the largest negative offset (closest midpoint *below* the cursor).
                     // Let's try finding the element whose top is closest *below* y.
                    return { offset: offset, element: child };
                } else {
                    return closest;
                }
                 // This logic finds the element whose midpoint is closest *above* the cursor 'y'.
                 // We need the element the dragged item should come *before*.

            }, { offset: Number.NEGATIVE_INFINITY }).element; // Start with negative infinity offset
        }


        // --- Component Initialization ---

        // Initialize components within a specific tab (e.g., when tab becomes active)
        function initializeTabComponents(tabElement) {
            if (!tabElement) { console.error("initializeTabComponents called with null element."); return; }
            const tabId = tabElement.dataset.tabId;
            console.log(`Initializing components for newly active tab: ${tabId}`);
            const activeView = tabElement.querySelector('.main-view-area > .view.active');
            if (activeView) {
                initializeViewComponents(activeView);
            } else {
                 console.warn(`No active view found within tab ${tabId} during initialization.`);
                 // Maybe find the default view for the tab and initialize that?
                 const defaultViewBtn = tabElement.querySelector('.view-switcher .view-btn.active') || tabElement.querySelector('.view-switcher .view-btn');
                 if (defaultViewBtn) {
                     const defaultViewId = defaultViewBtn.dataset.view;
                     const defaultView = tabElement.querySelector(`.main-view-area > .view[data-view-id="${defaultViewId}"]`);
                     if (defaultView) {
                          console.log(`Initializing default view ${defaultViewId} for tab ${tabId}`);
                          initializeViewComponents(defaultView);
                     }
                 }
            }
             // Special handling for Column view drag/drop setup - always needs listeners if present
             const columnView = tabElement.querySelector('.view[data-view-id="columns"]');
             if (columnView) {
                  console.log(`Ensuring drag/drop listeners are added to column view in tab ${tabId}`);
                  addDragDropListeners(columnView); // Add/Re-add listeners for column view
             }
        }

        // Initialize components within a specific view (e.g., slideshow, table)
        function initializeViewComponents(viewElement) {
             if (!viewElement) { console.error("initializeViewComponents called with null element."); return; }
             const viewType = viewElement.dataset.viewId;
             const tabId = viewElement.closest('.tab-content')?.dataset.tabId;
             console.log(`Initializing components for view: ${viewType} in tab ${tabId}`);
            if (viewType === 'slideshow') {
                 // Only initialize if not already done (check flag)
                 const container = viewElement.querySelector('.slideshow-container');
                 if (container && container.dataset.slideshowInitialized !== "true") {
                     initializeSlideshow(viewElement);
                 } else if (container) {
                     console.log(`Slideshow in tab ${tabId} already initialized.`);
                 }
            }
            else if (viewType === 'table') {
                 // Only initialize if not already done (check flag on headers)
                 const table = viewElement.querySelector('.data-table');
                 const needsInit = table && !table.querySelector('th[data-sortable][data-sorting-initialized="true"]');
                 if (needsInit) {
                     initializeTableSorting(viewElement);
                 } else if (table) {
                      console.log(`Table sorting in tab ${tabId} already initialized.`);
                 }
            }
             // Add initialization for other view-specific components if needed
        }

        // Slideshow specific initialization (attaching event listeners)
        function initializeSlideshow(slideshowView) {
             const container = slideshowView.querySelector('.slideshow-container');
             // Prevent double initialization
             if (!container || container.dataset.slideshowInitialized === "true") {
                 // console.log("Slideshow already initialized or container not found.");
                 return;
             }
             console.log("Initializing slideshow event listeners...");

            const slides = container.querySelectorAll('.slide');
            const prevBtn = container.querySelector('.slide-nav.prev');
            const nextBtn = container.querySelector('.slide-nav.next');

            if (slides.length === 0 || !prevBtn || !nextBtn) {
                 console.log(`Slideshow initialization skipped: ${slides.length} slides, prevBtn=${!!prevBtn}, nextBtn=${!!nextBtn}.`);
                 if (slides.length === 0 && !container.querySelector('.empty-view-message')) {
                      container.innerHTML = '<p class="empty-view-message">No items to display in slideshow.</p>';
                 }
                 container.dataset.slideshowInitialized = "true"; // Mark as initialized even if no slides
                return;
            }

            let currentSlideIndex = 0;

            function showSlide(index) {
                 // Wrap around logic
                 currentSlideIndex = (index + slides.length) % slides.length;
                slides.forEach((slide, i) => slide.classList.toggle('active', i === currentSlideIndex));
                 // console.log(`Slideshow showing slide ${currentSlideIndex}`); // Less verbose
            }

            // Remove old listeners before adding new ones (important if re-rendering)
            prevBtn.replaceWith(prevBtn.cloneNode(true)); // Clone to remove listeners
            nextBtn.replaceWith(nextBtn.cloneNode(true));
             // Re-select the cloned buttons
             const newPrevBtn = container.querySelector('.slide-nav.prev');
             const newNextBtn = container.querySelector('.slide-nav.next');

             if(newPrevBtn) newPrevBtn.addEventListener('click', () => showSlide(currentSlideIndex - 1));
             if(newNextBtn) newNextBtn.addEventListener('click', () => showSlide(currentSlideIndex + 1));

            showSlide(currentSlideIndex); // Show the first slide initially
            container.dataset.slideshowInitialized = "true"; // Mark as initialized
             console.log("Slideshow listeners initialized successfully.");
        }

        // Table sorting specific initialization
        function initializeTableSorting(tableView) {
            const table = tableView.querySelector('.data-table');
            if (!table) { console.warn("Data table not found for sorting initialization."); return; }
            console.log("Initializing table sorting listeners...");

            const headers = table.querySelectorAll('thead th[data-sortable]');

            headers.forEach(header => {
                 // Prevent double initialization
                 if (header.dataset.sortingInitialized === "true") return;

                 // Clone header to remove potential old listeners
                 const newHeader = header.cloneNode(true);
                 header.parentNode.replaceChild(newHeader, header);
                 newHeader.dataset.sortingInitialized = "true"; // Mark as initialized

                newHeader.addEventListener('click', () => {
                    const tbody = table.querySelector('tbody');
                    if (!tbody || tbody.children.length <= 1) return; // No sorting needed for 0 or 1 rows

                    const column = parseInt(newHeader.dataset.column || '-1', 10);
                    if (column < 0) return;

                    const currentDir = newHeader.dataset.sortDir;
                    const newDir = currentDir === 'asc' ? 'desc' : 'asc';
                    console.log(`Sorting table column ${column} ${newDir}`);

                    // Reset other headers' sort direction indicator
                    table.querySelectorAll('thead th[data-sortable]').forEach(h => {
                         if (h !== newHeader) {
                             delete h.dataset.sortDir;
                             h.querySelector('.sort-icon').textContent = ''; // Clear icon visually
                         }
                    });
                     // Update current header's direction indicator
                    newHeader.dataset.sortDir = newDir;
                    const icon = newHeader.querySelector('.sort-icon');
                    if(icon) icon.textContent = newDir === 'asc' ? '▲' : '▼'; // Use text content for icons


                    // Sort the rows
                    const rows = Array.from(tbody.querySelectorAll('tr'));
                    rows.sort((rowA, rowB) => {
                        // Robustly get cell text, default to empty string
                        const cellA = rowA.querySelectorAll('td')[column]?.textContent?.trim().toLowerCase() ?? '';
                        const cellB = rowB.querySelectorAll('td')[column]?.textContent?.trim().toLowerCase() ?? '';

                        // Basic locale-aware string comparison
                        const compare = cellA.localeCompare(cellB, undefined, {numeric: true, sensitivity: 'base'});

                        return newDir === 'asc' ? compare : -compare;
                    });

                    // Re-append rows in sorted order
                    rows.forEach(row => tbody.appendChild(row));
                });
            });
             console.log(`Table sorting listeners initialized for ${headers.length} headers.`);
        }


        // --- Initial Load ---
        async function loadDataAndRender() {
            console.log("Starting loadDataAndRender...");
            try {
                appData = await loadData(); // Load and sort data
                 if (!appData) throw new Error("loadData did not return valid data.");

                renderApp(); // Render all views based on loaded data
                console.log("Initial UI Rendered.");

                 // Ensure the correct initial tab is activated and its components initialized
                 const activeTabLink = document.querySelector('.tab-link.active');
                 let initialTabHtmlId = activeTabLink?.getAttribute('data-tab');
                 // If no tab is marked active in HTML, default to the first one
                 if (!initialTabHtmlId) {
                     const firstTabLink = document.querySelector('.tab-link');
                     if (firstTabLink) {
                          initialTabHtmlId = firstTabLink.getAttribute('data-tab');
                          console.log(`No active tab in HTML, defaulting to first tab: ${initialTabHtmlId}`);
                     }
                 }

                 if (initialTabHtmlId) {
                     // activateTab handles setting '.active' and calling initializeTabComponents
                     activateTab(initialTabHtmlId);
                 } else {
                     console.error("Could not determine initial tab to activate.");
                     alert("Error: Could not initialize application tabs.");
                 }

                 // Set initial sidebar state based on mode toggle
                const guidedModeInput = document.querySelector('input[name="mode"][value="guided"]');
                toggleSidebar(guidedModeInput?.checked ?? false);

                 console.log("loadDataAndRender finished successfully.");
            } catch (error) {
                console.error("FATAL Error during initial load and render:", error);
                alert("Failed to load initial application state. Please check the console and refresh the page.");
                 // Display error prominently if main structure fails
                 const mainElement = document.querySelector('main');
                 if(mainElement) mainElement.innerHTML = `<h2 class='error-message'>Application Error</h2><p>Could not load initial data. Please check the browser console (F12) for details and refresh the page.</p><p><i>Error: ${error.message}</i></p>`;
            }
        }

        // --- Global Error Handling ---
         window.addEventListener('error', function(event) {
             console.error('Global error caught:', event.error, event.message);
             // Optionally display a non-blocking notification to the user
         });
         window.addEventListener('unhandledrejection', function(event) {
             console.error('Unhandled promise rejection caught:', event.reason);
             // Optionally display a non-blocking notification
         });


        // Start the application
        loadDataAndRender();

    } catch (outerError) {
        // Catch errors during the initial synchronous setup phase
        console.error("A critical error occurred during script initialization:", outerError);
        alert("A critical error occurred. Please check the browser console (F12) and refresh the page.");
         const bodyElement = document.body;
         if(bodyElement) bodyElement.innerHTML = `<h2 style='color: red; margin: 20px;'>Application Initialization Error</h2><p style='margin: 20px;'>A critical error prevented the application from starting. Please check the browser console (F12) for details.</p><p style='margin: 20px;'><i>Error: ${outerError.message}</i></p>`;
    }
     console.log("End of script.js execution.");

}); // End DOMContentLoaded